import os
import re
import struct
import tkinter as tk
from tkinter import filedialog, messagebox, ttk
from PIL import Image, ImageTk
import numpy as np
import threading

class MipmappedTexConverter:
    def __init__(self, root):
        self.root = root
        self.root.title("Mipmapped TEX Converter (M0/M1)")
        self.root.geometry("800x650")
        
        # Store original headers for reimport
        self.original_headers = {}
        
        # Set up the main frame
        self.main_frame = ttk.Frame(root, padding=10)
        self.main_frame.pack(fill=tk.BOTH, expand=True)
        
        # Path entry
        ttk.Label(self.main_frame, text="Folder with .tex files:").grid(row=0, column=0, sticky=tk.W)
        self.folder_path = tk.StringVar()
        self.path_entry = ttk.Entry(self.main_frame, width=50, textvariable=self.folder_path)
        self.path_entry.grid(row=0, column=1, sticky=(tk.W, tk.E), padx=5)
        ttk.Button(self.main_frame, text="Browse...", command=self.browse_folder).grid(row=0, column=2, padx=5)
        
        # Export frame
        export_frame = ttk.LabelFrame(self.main_frame, text="Export Options", padding=10)
        export_frame.grid(row=1, column=0, columnspan=3, sticky=(tk.W, tk.E), pady=10)
        
        ttk.Button(export_frame, text="Export All Textures", 
                  command=lambda: self.export_textures(select_files=False)).grid(row=0, column=0, padx=5, pady=5, sticky=(tk.W, tk.E))
        
        ttk.Button(export_frame, text="Export Selected Textures", 
                  command=lambda: self.export_textures(select_files=True)).grid(row=0, column=1, padx=5, pady=5, sticky=(tk.W, tk.E))
        
        # Import frame
        import_frame = ttk.LabelFrame(self.main_frame, text="Import Options", padding=10)
        import_frame.grid(row=2, column=0, columnspan=3, sticky=(tk.W, tk.E), pady=10)
        
        ttk.Button(import_frame, text="Import All Textures", 
                  command=lambda: self.import_textures(select_files=False)).grid(row=0, column=0, padx=5, pady=5, sticky=(tk.W, tk.E))
        
        ttk.Button(import_frame, text="Import Selected Textures", 
                  command=lambda: self.import_textures(select_files=True)).grid(row=0, column=1, padx=5, pady=5, sticky=(tk.W, tk.E))
        
        # Settings frame
        settings_frame = ttk.LabelFrame(self.main_frame, text="Settings", padding=10)
        settings_frame.grid(row=3, column=0, columnspan=3, sticky=(tk.W, tk.E), pady=10)
        
        ttk.Label(settings_frame, text="Export Format:").grid(row=0, column=0, sticky=tk.W)
        self.export_format = tk.StringVar(value="auto")
        ttk.Radiobutton(settings_frame, text="Automatic (M0=16bit, M1=32bit)", 
                        variable=self.export_format, value="auto").grid(row=0, column=1, sticky=tk.W)
        ttk.Radiobutton(settings_frame, text="Force All to 32-bit TGA", 
                        variable=self.export_format, value="32bit").grid(row=1, column=1, sticky=tk.W)
        
        # Status and log
        status_frame = ttk.LabelFrame(self.main_frame, text="Status", padding=10)
        status_frame.grid(row=4, column=0, columnspan=3, sticky=(tk.W, tk.E, tk.N, tk.S), pady=10)
        
        # Progress bar
        self.progress_var = tk.DoubleVar()
        self.progress_bar = ttk.Progressbar(status_frame, variable=self.progress_var, maximum=100)
        self.progress_bar.grid(row=0, column=0, columnspan=3, sticky=(tk.W, tk.E), pady=5)
        
        # Log box
        self.log_box = tk.Text(status_frame, height=10, width=80, wrap=tk.WORD)
        self.log_box.grid(row=1, column=0, columnspan=3, sticky=(tk.W, tk.E, tk.N, tk.S), pady=5)
        
        # Add scrollbar
        scrollbar = ttk.Scrollbar(status_frame, orient="vertical", command=self.log_box.yview)
        scrollbar.grid(row=1, column=3, sticky=(tk.N, tk.S))
        self.log_box.configure(yscrollcommand=scrollbar.set)
        
        # Info box
        info_frame = ttk.LabelFrame(self.main_frame, text="Information", padding=10)
        info_frame.grid(row=5, column=0, columnspan=3, sticky=(tk.W, tk.E), pady=10)
        
        info_text = """This tool handles mipmapped textures for your game:

• Non-Alpha Textures: Exported as M_0 (512x512) at 16-bit format
• Alpha Textures: Exported as M_1 (256x256) at 32-bit format to preserve transparency

When editing in your image editor, do not change the dimensions or format.
This approach allows you to have full alpha transparency in the game by using M_1 textures."""
        
        info_label = ttk.Label(info_frame, text=info_text, justify=tk.LEFT, wraplength=750)
        info_label.grid(row=0, column=0, sticky=(tk.W, tk.E))
        
        # Status bar
        self.status_var = tk.StringVar()
        self.status_var.set("Ready")
        status_bar = ttk.Label(root, textvariable=self.status_var, relief=tk.SUNKEN, anchor=tk.W)
        status_bar.pack(side=tk.BOTTOM, fill=tk.X)
        
        # Initialize log
        self.log("Program initialized. Please select a folder with .tex files.")
    
    def browse_folder(self):
        folder = filedialog.askdirectory(title="Select Folder with .tex files")
        if folder:
            self.folder_path.set(folder)
            self.log(f"Selected folder: {folder}")
    
    def log(self, message):
        """Add a message to the log box"""
        self.log_box.insert(tk.END, message + "\n")
        self.log_box.see(tk.END)  # Scroll to end
        self.status_var.set(message)
        self.root.update_idletasks()
        print(message)  # Also print to console
    
    def update_progress(self, value):
        """Update progress bar"""
        self.progress_var.set(value)
        self.root.update_idletasks()
    
    def export_textures(self, select_files=False):
        """Export TEX files to TGA with correct mipmapping"""
        folder = self.folder_path.get()
        if not folder or not os.path.isdir(folder):
            messagebox.showerror("Error", "Please select a valid folder")
            return
        
        output_folder = os.path.join(folder, "textures2")
        if not os.path.exists(output_folder):
            os.makedirs(output_folder)
            
        # Create M_0 and M_1 folders
        m0_folder = os.path.join(output_folder, "m_0")
        m1_folder = os.path.join(output_folder, "m_1")
        
        if not os.path.exists(m0_folder):
            os.makedirs(m0_folder)
        if not os.path.exists(m1_folder):
            os.makedirs(m1_folder)
        
        # Get files to process
        files_to_process = []
        if select_files:
            files = filedialog.askopenfilenames(
                title="Select TEX files to export",
                filetypes=[("TEX files", "*.tex")],
                initialdir=folder
            )
            if not files:
                self.log("No files selected")
                return
            
            # Extract just filenames
            files_to_process = [os.path.basename(f) for f in files]
        else:
            # Get all TEX files
            files_to_process = [f for f in os.listdir(folder) if f.lower().endswith('.tex')]
        
        if not files_to_process:
            self.log("No TEX files found in the selected folder")
            return
        
        self.log(f"Exporting {len(files_to_process)} TEX files...")
        
        # Start a thread for processing
        threading.Thread(target=self._export_thread, args=(folder, files_to_process, m0_folder, m1_folder)).start()
    
    def _export_thread(self, folder, files_to_process, m0_folder, m1_folder):
        """Thread function for exporting textures"""
        total_files = len(files_to_process)
        processed = 0
        
        for file in files_to_process:
            # Update progress
            self.update_progress((processed / total_files) * 100)
            processed += 1
            
            try:
                input_path = os.path.join(folder, file)
                basename = os.path.splitext(file)[0]
                
                # Remove any existing mipmap suffix for clean base name
                clean_name = re.sub(r"_m_\d+$", "", basename)
                
                # Check if it's an alpha texture
                is_alpha = "alpha" in file.lower()
                
                # Read TEX file
                with open(input_path, 'rb') as f:
                    data = f.read()
                
                if len(data) < 32:
                    self.log(f"Error: {file} is too small to be a valid texture")
                    continue
                
                # Store original header for later
                self.original_headers[file.lower()] = data[:32]
                
                # Parse header
                width = struct.unpack_from("<I", data, 20)[0]
                height = struct.unpack_from("<I", data, 24)[0]
                bpp = struct.unpack_from("<I", data, 28)[0]
                
                # Pixel data
                pixels = data[32:]
                
                # Create PIL image from texture data
                if bpp == 16:
                    # 16-bit format
                    img = self._create_image_from_16bit(pixels, width, height)
                elif bpp == 32:
                    # 32-bit format
                    img = self._create_image_from_32bit(pixels, width, height)
                elif bpp == 24:
                    # 24-bit format
                    img = self._create_image_from_24bit(pixels, width, height)
                else:
                    self.log(f"Unsupported format: {bpp} bpp for {file}")
                    continue
                
                # Determine output location
                if is_alpha:
                    # Alpha textures go to M_1 (256x256)
                    output_folder = m1_folder
                    output_name = f"{clean_name}_m_1.tga"
                    target_size = (256, 256)
                else:
                    # Regular textures go to M_0 (512x512)
                    output_folder = m0_folder
                    output_name = f"{clean_name}_m_0.tga"
                    target_size = (512, 512)
                
                # Resize if needed
                if img.size != target_size:
                    img = img.resize(target_size, Image.LANCZOS)
                
                # Set output path
                output_path = os.path.join(output_folder, output_name)
                
                # Save as TGA
                self._save_tga(img, output_path, force_32bit=(is_alpha or self.export_format.get() == "32bit"))
                
                self.log(f"Exported {file} to {output_name}")
                
            except Exception as e:
                self.log(f"Error processing {file}: {str(e)}")
        
        # Update progress to 100%
        self.update_progress(100)
        self.log(f"Export completed: {processed} files processed")
        messagebox.showinfo("Export Complete", f"Successfully exported {processed} files.")
    
    def _create_image_from_16bit(self, pixel_data, width, height):
        """Create a PIL image from 16-bit TEX data"""
        rgba = bytearray(width * height * 4)
        idx = 0
        
        for i in range(0, len(pixel_data), 2):
            if i+1 >= len(pixel_data):
                break
                
            val = pixel_data[i] | (pixel_data[i+1] << 8)
            
            # 16-bit format: rrrrrgggggabbbbb (read documentation)
            blue = val & 0x1F
            attr = (val >> 5) & 0x1  # Attribute/alpha bit (1 = transparent)
            green = (val >> 6) & 0x1F
            red = (val >> 11) & 0x1F
            
            # Convert to 8-bit per channel
            r8 = (red * 255) // 31
            g8 = (green * 255) // 31
            b8 = (blue * 255) // 31
            a8 = 0 if attr == 1 else 255  # 0 = transparent, 255 = opaque
            
            rgba[idx] = r8
            rgba[idx+1] = g8
            rgba[idx+2] = b8
            rgba[idx+3] = a8
            idx += 4
        
        return Image.frombytes("RGBA", (width, height), bytes(rgba), "raw", "RGBA", 0, 1)
    
    def _create_image_from_24bit(self, pixel_data, width, height):
        """Create a PIL image from 24-bit TEX data"""
        rgb = bytearray(width * height * 3)
        
        for i in range(0, len(pixel_data), 3):
            if i+2 >= len(pixel_data):
                break
                
            # TEX format is BGR order
            rgb[i] = pixel_data[i+2]  # R from B
            rgb[i+1] = pixel_data[i+1]  # G
            rgb[i+2] = pixel_data[i]  # B from R
        
        img = Image.frombytes("RGB", (width, height), bytes(rgb), "raw", "RGB", 0, 1)
        return img.convert("RGBA")  # Add alpha channel
    
    def _create_image_from_32bit(self, pixel_data, width, height):
        """Create a PIL image from 32-bit TEX data"""
        rgba = bytearray(width * height * 4)
        
        for i in range(0, len(pixel_data), 4):
            if i+3 >= len(pixel_data):
                break
                
            # TEX format is BGRA order
            rgba[i] = pixel_data[i+2]  # R from B
            rgba[i+1] = pixel_data[i+1]  # G
            rgba[i+2] = pixel_data[i]  # B from R
            rgba[i+3] = pixel_data[i+3]  # A
        
        return Image.frombytes("RGBA", (width, height), bytes(rgba), "raw", "RGBA", 0, 1)
    
    def _save_tga(self, img, output_path, force_32bit=False):
        """Save a PIL image as TGA format"""
        # Make sure image is RGBA
        if img.mode != "RGBA":
            img = img.convert("RGBA")
        
        width, height = img.size
        
        # Create TGA header
        header = bytearray(18)
        header[2] = 2  # Uncompressed RGB
        header[12] = width & 0xFF
        header[13] = (width >> 8) & 0xFF
        header[14] = height & 0xFF
        header[15] = (height >> 8) & 0xFF
        
        # Always use 32-bit for alpha textures or if forced
        if force_32bit:
            header[16] = 32  # 32-bit
            header[17] = 0x28  # 8 bits alpha + top-down
            
            # Get raw pixel data
            rgba = img.tobytes("raw", "RGBA", 0, 1)
            
            # Convert to BGRA for TGA
            bgra = bytearray(len(rgba))
            for i in range(0, len(rgba), 4):
                bgra[i] = rgba[i+2]  # B from R
                bgra[i+1] = rgba[i+1]  # G
                bgra[i+2] = rgba[i]  # R from B
                bgra[i+3] = rgba[i+3]  # A
            
            # Write TGA file
            with open(output_path, "wb") as f:
                f.write(header)
                f.write(bgra)
        else:
            # Use 16-bit format
            header[16] = 16  # 16-bit
            header[17] = 0x21  # 1 bit alpha + top-down
            
            # Get raw pixel data
            rgba = img.tobytes("raw", "RGBA", 0, 1)
            
            # Convert to 16-bit format for TGA
            data16 = bytearray(width * height * 2)
            idx = 0
            
            for i in range(0, len(rgba), 4):
                r = rgba[i]
                g = rgba[i+1]
                b = rgba[i+2]
                a = rgba[i+3]
                
                # Convert to 5-5-5-1 format
                r5 = (r * 31) // 255
                g5 = (g * 31) // 255
                b5 = (b * 31) // 255
                a1 = 1 if a >= 128 else 0
                
                # TGA format: bits 0..4=blue, 5..9=green, 10..14=red, 15=alpha
                val = (b5) | (g5 << 5) | (r5 << 10) | (a1 << 15)
                
                data16[idx] = val & 0xFF
                data16[idx+1] = (val >> 8) & 0xFF
                idx += 2
            
            # Write TGA file
            with open(output_path, "wb") as f:
                f.write(header)
                f.write(data16)
    
    def import_textures(self, select_files=False):
        """Import TGA files back to TEX format"""
        folder = self.folder_path.get()
        if not folder or not os.path.isdir(folder):
            messagebox.showerror("Error", "Please select a valid folder")
            return
        
        # Get TGA files to import
        files_to_process = []
        
        # Look in m_0 and m_1 folders
        m0_folder = os.path.join(folder, "textures2", "m_0")
        m1_folder = os.path.join(folder, "textures2", "m_1")
        
        if not os.path.exists(m0_folder) and not os.path.exists(m1_folder):
            messagebox.showerror("Error", "Export textures first - no m_0 or m_1 folders found")
            return
        
        if select_files:
            # Let user select files from both folders
            all_files = []
            if os.path.exists(m0_folder):
                all_files.extend([os.path.join(m0_folder, f) for f in os.listdir(m0_folder) if f.lower().endswith('.tga')])
            if os.path.exists(m1_folder):
                all_files.extend([os.path.join(m1_folder, f) for f in os.listdir(m1_folder) if f.lower().endswith('.tga')])
            
            if not all_files:
                messagebox.showinfo("Info", "No TGA files found in m_0 or m_1 folders")
                return
            
            # Let user select files
            selected_files = filedialog.askopenfilenames(
                title="Select TGA files to import",
                filetypes=[("TGA files", "*.tga")],
                initialdir=os.path.dirname(all_files[0])
            )
            
            if not selected_files:
                self.log("No files selected")
                return
            
            # Process selected files
            for file in selected_files:
                # Store (folder, filename) tuple
                parent_folder = os.path.dirname(file)
                filename = os.path.basename(file)
                files_to_process.append((parent_folder, filename))
        else:
            # Process all files in both folders
            if os.path.exists(m0_folder):
                for file in os.listdir(m0_folder):
                    if file.lower().endswith('.tga'):
                        files_to_process.append((m0_folder, file))
            
            if os.path.exists(m1_folder):
                for file in os.listdir(m1_folder):
                    if file.lower().endswith('.tga'):
                        files_to_process.append((m1_folder, file))
        
        if not files_to_process:
            self.log("No TGA files found in the mipmap folders")
            return
        
        self.log(f"Importing {len(files_to_process)} TGA files...")
        
        # Start a thread for processing
        threading.Thread(target=self._import_thread, args=(folder, files_to_process)).start()
    
    def _import_thread(self, folder, files_to_process):
        """Thread function for importing textures"""
        total_files = len(files_to_process)
        processed = 0
        
        for source_folder, file in files_to_process:
            # Update progress
            self.update_progress((processed / total_files) * 100)
            processed += 1
            
            try:
                input_path = os.path.join(source_folder, file)
                basename = os.path.splitext(file)[0]
                
                # Output goes to the main folder
                output_path = os.path.join(folder, basename + '.tex')
                
                # Check if it's an M_1 texture (alpha)
                is_m1 = "_m_1" in file.lower()
                is_alpha = "alpha" in file.lower()
                
                # Read TGA file
                with open(input_path, 'rb') as f:
                    data = f.read()
                
                if len(data) < 18:
                    self.log(f"Error: {file} is too small to be a valid TGA")
                    continue
                
                # Parse TGA header
                tga_type = data[2]
                width = (data[13] << 8) | data[12]
                height = (data[15] << 8) | data[14]
                bpp = data[16]
                descriptor = data[17]
                
                # Pixel data
                pixels = data[18:]
                
                if tga_type != 2:
                    self.log(f"Error: {file} is not an uncompressed TGA")
                    continue
                
                # Create PIL image from TGA data
                if bpp == 32:
                    # 32-bit TGA to RGBA
                    img = Image.frombytes("RGBA", (width, height), pixels, "raw", "BGRA", 0, 1)
                elif bpp == 24:
                    # 24-bit TGA to RGB
                    img = Image.frombytes("RGB", (width, height), pixels, "raw", "BGR", 0, 1)
                    img = img.convert("RGBA")  # Add alpha channel
                elif bpp == 16:
                    # 16-bit TGA to RGBA
                    rgba = bytearray(width * height * 4)
                    idx = 0
                    
                    for i in range(0, len(pixels), 2):
                        if i+1 >= len(pixels):
                            break
                            
                        val = pixels[i] | (pixels[i+1] << 8)
                        
                        # TGA format: bits 0..4=blue, 5..9=green, 10..14=red, 15=alpha
                        blue = val & 0x1F
                        green = (val >> 5) & 0x1F
                        red = (val >> 10) & 0x1F
                        alpha = (val >> 15) & 0x1
                        
                        # Convert to 8-bit
                        r8 = (red * 255) // 31
                        g8 = (green * 255) // 31
                        b8 = (blue * 255) // 31
                        a8 = 255 if alpha else 0
                        
                        rgba[idx] = r8
                        rgba[idx+1] = g8
                        rgba[idx+2] = b8
                        rgba[idx+3] = a8
                        idx += 4
                    
                    img = Image.frombytes("RGBA", (width, height), bytes(rgba), "raw", "RGBA", 0, 1)
                else:
                    self.log(f"Unsupported TGA format: {bpp} bpp")
                    continue
                
                # Create TEX file
                if is_m1 or is_alpha:
                    # M_1 textures use 32-bit format
                    tex_data = self._create_32bit_tex(img, width, height)
                else:
                    # M_0 textures use 16-bit format
                    tex_data = self._create_16bit_tex(img, width, height)
                
                # Write TEX file
                with open(output_path, "wb") as f:
                    f.write(tex_data)
                
                self.log(f"Imported {file} to {os.path.basename(output_path)}")
                
            except Exception as e:
                self.log(f"Error importing {file}: {str(e)}")
        
        # Update progress to 100%
        self.update_progress(100)
        self.log(f"Import completed: {processed} files processed")
        messagebox.showinfo("Import Complete", f"Successfully imported {processed} files.")
    
    def _create_16bit_tex(self, img, width, height):
        """Create a 16-bit TEX file from a PIL image"""
        # Create header
        header = bytearray(32)
        
        # Version 2
        struct.pack_into("<I", header, 0, 2)
        
        # Float 16.0
        header[4] = 0x00
        header[5] = 0x00
        header[6] = 0x80
        header[7] = 0x41
        
        # Other standard values
        struct.pack_into("<I", header, 8, 1)
        struct.pack_into("<I", header, 16, 1)
        struct.pack_into("<I", header, 20, width)
        struct.pack_into("<I", header, 24, height)
        struct.pack_into("<I", header, 28, 16)  # 16-bit
        
        # Create pixel data
        pixel_data = bytearray(width * height * 2)
        img_data = img.tobytes("raw", "RGBA", 0, 1)
        
        idx = 0
        for i in range(0, len(img_data), 4):
            r = img_data[i]
            g = img_data[i+1]
            b = img_data[i+2]
            a = img_data[i+3]
            
            # Convert to 5-5-5-1 format
            r5 = (r * 31) // 255
            g5 = (g * 31) // 255
            b5 = (b * 31) // 255
            
            # Attribute bit: 1 = transparent, 0 = opaque
            attr = 1 if a < 128 else 0
            
            # TEX format: bits 0..4=blue, 5=attr, 6..10=green, 11..15=red
            val = (b5) | (attr << 5) | (g5 << 6) | (r5 << 11)
            
            pixel_data[idx] = val & 0xFF
            pixel_data[idx+1] = (val >> 8) & 0xFF
            idx += 2
        
        return header + pixel_data
    
    def _create_32bit_tex(self, img, width, height):
        """Create a 32-bit TEX file from a PIL image"""
        # Create header
        header = bytearray(32)
        
        # Version 2
        struct.pack_into("<I", header, 0, 2)
        
        # Float 16.0
        header[4] = 0x00
        header[5] = 0x00
        header[6] = 0x80
        header[7] = 0x41
        
        # Other standard values
        struct.pack_into("<I", header, 8, 1)
        header[9] = 0x01  # Extra flag for M_1
        struct.pack_into("<I", header, 16, 1)
        struct.pack_into("<I", header, 20, width)
        struct.pack_into("<I", header, 24, height)
        struct.pack_into("<I", header, 28, 32)  # 32-bit
        
        # Create pixel data
        pixel_data = bytearray(width * height * 4)
        img_data = img.tobytes("raw", "RGBA", 0, 1)
        
        for i in range(0, len(img_data), 4):
            # TEX format uses BGRA order
            pixel_data[i] = img_data[i+2]  # B from R
            pixel_data[i+1] = img_data[i+1]  # G
            pixel_data[i+2] = img_data[i]  # R from B
            pixel_data[i+3] = img_data[i+3]  # A
        
        return header + pixel_data

def main():
    root = tk.Tk()
    app = MipmappedTexConverter(root)
    root.mainloop()

if __name__ == "__main__":
    main()